-- Multi-Edit tab in MassFX Tools dialog.
-- Framework for Multi-Edit system, and rollouts for Rigid Body objects
-- Properties tab in MassFX Tools dialog
-- UI Scripted by Sergio Mucino. 10.4.2010.
-- Functionality scripted by Michaelson Britt. 10.5.2010.


filein "px_panelMultiEditor_Constraints.ms"

---------- ---------- ---------- ----------
-- Forward declarations
---------- ---------- ---------- ----------

global px_multiedit_methods  -- forward declaration


---------- ---------- ---------- ----------
-- Helpers
---------- ---------- ---------- ----------

struct KeyVal_struct
(
	keyVal = undefined,
	fn comparator item1 item2 =
	(
		case of
		(
		(item1.keyVal < item2.keyVal): -1
		(item1.keyVal > item2.keyVal): 1
		default: 0
		)
	)
)

struct ControlValLookup_struct
(
	ControlToParamFn,	-- Takes two parameters, controlItem and controlVal, returns paramVal
	ParamToControlFn	-- Takes two parameters, controlItem and paramVal, returns controlVal
)

-- Defined in px_globals.ms:
-- struct MultiEdit_ControlInfo_struct


---------- ---------- ---------- ----------
-- PromptForName helper
-- Invoke a prompt for name dialog
-- Usage is (name = PromptForNameHandler.PromptForName title initialName)
---------- ---------- ---------- ----------

struct PromptForNameHandler_struct
(
	resultNameVal, initNameVal, initTitleVal,
	PromptForNameRollout,
	
	fn PromptForName titleVal nameVal =
	(
		initTitleVal = titleVal
		initNameVal = nameVal
		CreateDialog PromptForNameRollout modal:true
		resultNameVal
	)
)
PromptForNameHandler = PromptForNameHandler_struct()

rollout PromptForNameRollout "Prompt For Name" width:310 height:111 
(
	button btn_ok "OK" pos:[138,75] width:74 height:23
	button btn_cancel "Cancel" pos:[223,76] width:74 height:23
	editText edt_name "" pos:[8,33] width:288

	on btn_ok pressed do
	(
		if (edt_name.text.count > 128 or edt_name.text.count == 0) then
		(
			PromptForNameHandler.resultNameVal = undefined
			messageBox nvpxText.TXT_ERROR_NAME_SIZE
		)
		else
		(
			PromptForNameHandler.resultNameVal = edt_name.text
			DestroyDialog PromptForNameRollout
		)
	)

	on btn_cancel pressed do
	(
		PromptForNameHandler.resultNameVal = undefined
		DestroyDialog PromptForNameRollout	
	)

	on PromptForNameRollout open do
	(
		PromptForNameRollout.title = PromptForNameHandler.initTitleVal
		edt_name.text = PromptForNameHandler.initNameVal
		btn_ok.text = nvpxText.panelMultiOk
		btn_cancel.text = nvpxText.panelMultiCancel
	)
)
PromptForNameHandler.PromptForNameRollout = PromptForNameRollout


---------- ---------- ---------- ----------
-- Selection Rollout
---------- ---------- ---------- ----------

-- TO DO: use nvpxText
rollout px_panel_multiedit_selection "Selection" rolledUp:false category:1
(
	label lbl_Selection1 "" align:#center
	label lbl_Selection2 "" align:#center
	label lbl_Selection3 "" align:#center
	label lbl_Selection4 "" align:#center
	
	fn IsSelectionEmpty =
	(
		(pxSelectedRigidBodyMods.count==0) and (pxSelectedConstraints.count==0)
	)
	
	fn IsSelectionMixed =
	(
		(pxSelectedRigidBodyMods.count>0) and (pxSelectedConstraints.count>0)
	)
	
	fn IsVisible =
	(
		(px_multiedit_methods.IsTabVisible()) and 
			(IsSelectionEmpty()) or (IsSelectionMixed())
	)
	
	local controlInfoList =
	#(
	)
	
	fn UpdateUI =
	(
		px_panel_multiedit_selection.title = nvpxText.panelMultiSelectionTitle
		if (IsSelectionEmpty()) do
		(
			lbl_Selection1.text = nvpxText.panelMultiSelectionEmpty_Part1
			lbl_Selection2.text = nvpxText.panelMultiSelectionEmpty_Part2
			lbl_Selection3.text = nvpxText.panelMultiSelectionEmpty_Part3
			lbl_Selection4.text = nvpxText.panelMultiSelectionEmpty_Part4
		)
		if (IsSelectionMixed()) do
		(
			lbl_Selection1.text = nvpxText.panelMultiSelectionMixed_Part1
			lbl_Selection2.text = nvpxText.panelMultiSelectionMixed_Part2
			lbl_Selection3.text = nvpxText.panelMultiSelectionMixed_Part3
			lbl_Selection4.text = nvpxText.panelMultiSelectionMixed_Part4
		)
	)
)

---------- ---------- ---------- ----------
-- Properties Rollout
---------- ---------- ---------- ----------

rollout px_panel_multiedit_properties "Rigid Body Properties" rolledUp:false category:2
(

	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC

	label lbl_RigidBodyType "Rigid Body Type" across:2 offset:[0,4]
	dropdownList ddl_RigidBodyType "" across:1 width:70 align:#right items:#("Dynamic", "Kinematic", "Static" )
	checkbox chk_UntilFrame "Until Frame" across:2 enabled:false
	spinner spn_StartFrame "" width:70 offset:[-17,0] across:1 enabled:false type:#integer range:[0, 1000000, 0]
	button btn_Bake "Bake" across:1 width:150 offset:[-2,6]
	checkbox chk_UseHVC "Use High Velocity Collisions" offset:[0,6] across:1 align:#left 
	checkbox chk_StartSM "Start in Sleep Mode" across:1 align:#left
	checkbox chk_collideRB "Collide with Rigid Bodies" across:1 align:#left checked:true

	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC

	struct RigidBodyTypeLookup_struct
	(
		fn ControlToParamFn controlItem controlVal = ( controlVal ),
		fn ParamToControlFn controlItem paramVal =   ( paramVal )
	)
	local rigidBodyTypeLookup = RigidBodyTypeLookup_struct()

	local controlInfoList =
	#(
		(MultiEdit_ControlInfo_struct		controlItem:lbl_RigidBodyType),
		(MultiEdit_ControlInfo_struct		controlItem:ddl_RigidBodyType		paramIdent:#type  controlValLookup:rigidBodyTypeLookup ),
		(MultiEdit_ControlInfo_struct		controlItem:chk_UntilFrame			paramIdent:#switchType),
		(MultiEdit_ControlInfo_struct		controlItem:spn_StartFrame			paramIdent:#switchTypeAtFrame),
		(MultiEdit_ControlInfo_struct		controlItem:btn_Bake				paramIdent:#baked),
		(MultiEdit_ControlInfo_struct		controlItem:chk_UseHVC				paramIdent:#continuousCollisionDetection),
		(MultiEdit_ControlInfo_struct		controlItem:chk_StartSM				paramIdent:#sleepAtStart),
		(MultiEdit_ControlInfo_struct		controlItem:chk_collideRB			paramIdent:#CollideWithRigidBodies)
	)

	on chk_UntilFrame changed val do		px_multiedit_methods.SetControlToRigidBodyList chk_UntilFrame val
	on spn_StartFrame changed val do		px_multiedit_methods.SetControlToRigidBodyList spn_StartFrame val
	on chk_UseHVC changed val do			px_multiedit_methods.SetControlToRigidBodyList chk_UseHVC val
	on chk_StartSM changed val do			px_multiedit_methods.SetControlToRigidBodyList chk_StartSM val
	on chk_collideRB changed val do			px_multiedit_methods.SetControlToRigidBodyList chk_collideRB val

	fn IsVisible =
	(
		(px_multiedit_methods.IsTabVisible()) and
			px_multiedit_methods.showRigidBodyRollouts
	)
	
	fn UpdateUI =
	(
		-- Basic default handling
		px_multiedit_methods.UpdateUI_DefaultHandling_RigidBody controlInfoList updateEnables:false

		-- set title for the Bake button
		local bakedVal = px_multiedit_methods.GetParamFromRigidBodyList #baked
		if (bakedVal!=undefined) do bakedVal = (bakedVal>0)  -- Bake modes other than zero indicate baked object

		if bakedVal==false do		btn_Bake.text = nvpxText.panelMultiRBBake
		if bakedVal==true do		btn_Bake.text = nvpxText.panelMultiRBUnbake
		if bakedVal==undefined do	btn_Bake.text = nvpxText.panelMultiRBBakeUnbake

		-- Enable controls if none of the selected objects are baked
		ddl_RigidBodyType.enabled = (bakedVal==false)
		chk_UseHVC.enabled = (bakedVal==false)
		chk_StartSM.enabled = (bakedVal==false)
		chk_collideRB.enabled = (bakedVal==false)

		-- enable/disable the Until Frame UI components
		local enableUntilFrame = ((bakedVal==false) and (ddl_RigidBodyType.selection==2))
		chk_untilFrame.enabled = spn_startFrame.enabled = enableUntilFrame
	)

	on ddl_RigidBodyType selected val do (
		px_multiedit_methods.SetControlToRigidBodyList ddl_RigidBodyType val

		UpdateUI()
		px_multiedit_methods.UpdateUI_MeshRollout()
	)

	on btn_Bake pressed do
	(
		local bakedVal = px_multiedit_methods.GetParamFromRigidBodyList #baked
		if (bakedVal!=undefined) do bakedVal = (bakedVal>0)  -- Bake modes other than zero indicate baked object

		if bakedVal==false do	pxBakeSelectionUndoOn false  -- Bake, "remove" is false
		if bakedVal==true do	pxBakeSelectionUndoOn true	-- Unbake, "remove" is true
		UpdateUI()
	)

	on px_panel_multiedit_properties open do
	(
		px_panel_multiedit_properties.title = nvpxText.panelMultiRB
		lbl_RigidBodyType.text = nvpxText.panelMultiRBType
		ddl_RigidBodyType.items = #( nvpxText.panelMultiRBTypeDynamic, nvpxText.panelMultiRBTypeKinematic, nvpxText.panelMultiRBTypeStatic )
		chk_UntilFrame.text = nvpxText.panelMultiRBUntilFrame
		btn_Bake.text = nvpxText.panelMultiRBBake
		chk_UseHVC.text = nvpxText.panelMultiRBHVC
		chk_StartSM.text = nvpxText.panelMultiRBSleep
		chk_collideRB.text = nvpxText.panelMultiRBCollide
	)

	on px_panel_multiedit_properties close do
	(
	)

	on px_panel_multiedit_properties help do (HelpSystem.ShowProductHelp 15012) --id_massfx_tools_edit

)


---------- ---------- ---------- ----------
-- Material List Rollout
---------- ---------- ---------- ----------

rollout px_panel_multiedit_materialList "Physical Material" rolledUp:false category:3
(
	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC
	
	label lbl_presetMatList "Preset" pos:[8,11] width:40 height:16
	dropdownList ddl_presetMatList "" pos:[48,8] width:95 height:21
	checkbutton btn_pickMat "P" pos:[148,6] width:24 height:24 iconName:"MaterialEditor/PickMaterial"
	button btn_createPreset "Create Preset" pos:[6,42] width:80
	button btn_deletePreset "Delete Preset" pos:[92,42] width:80

	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC

	-- TO DO: Define controlValMapping for ddl_presetMatList, so it can use the default handling
	
	local controlInfoList = #(
		(MultiEdit_ControlInfo_struct		controlItem:lbl_presetMatList),
		(MultiEdit_ControlInfo_struct		controlItem:ddl_presetMatList),
		(MultiEdit_ControlInfo_struct		controlItem:btn_pickMat),
		(MultiEdit_ControlInfo_struct		controlItem:btn_createPreset),
		(MultiEdit_ControlInfo_struct		controlItem:btn_deletePreset)
	)

	local paramRollout -- Parameter rollout is driven as a slave of this rollout
	local presetMatIdList -- Preset Material ID values for items in the dropdown
	local presetMatIdSel -- Current Material ID value for item in the dropdown
	local presetMatIndexSel -- Current Material ID value for item in the dropdown

	fn IsVisible =
	(
		(px_multiedit_methods.IsTabVisible()) and
			px_multiedit_methods.showRigidBodyRollouts
	)

	fn UpdateUI_PresetMatList isForceUpdate:false =
	(
		-- Determine the selected preset
		presetMatIdSel = px_multiedit_methods.GetParamFromRigidBodyList #materialId
		presetMatIndexSel = 0

		-- Create the lists of preset name and id values
		if isForceUpdate do presetMatIdList = #(0, undefined) -- Material ID zero corresponds to the "none" entry
		local presetMatNameList = #(nvpxText.panelMultiMatPresetNone,"---------") -- TO DO: Localize this

		local i, matIdCount = nvpx.MaterialGetCount()
		for i = 1 to matIdCount do
		(
			matId = nvpx.MaterialGetId (i-1)  -- TO DO: Method should use 1-based indexing, not 0-based
			if isForceUpdate do append presetMatIdList matId
			if isForceUpdate do append presetMatNameList (nvpx.MaterialGetParam matId "Name")
			-- Add 2, account for first two entries, "(none)" and "---------"
			if presetMatIdSel==matId do presetMatIndexSel = (i+2)
		)
		if presetMatIdSel==0 do presetMatIndexSel=1 -- Special case: First dropdown entry for the "none" preset

		-- Populate the preset dropdown
		ddl_presetMatList.selection = presetMatIndexSel
		if isForceUpdate do ddl_presetMatList.items = presetMatNameList
	)

	fn UpdateUI_ParamRolloutEnables isForceUpdate:false =
	(
		if paramRollout!=undefined do
		(
			-- Get the volume, if all objects agree, or undefined if they disagree
			local volumeVal = (px_multiedit_methods.GetParamFromRigidBodyList #volume)

			local isPresetAgree = (presetMatIdSel != undefined)
			local isBasePreset = (isPresetAgree and (presetMatIdSel==0))
			local isNonBasePreset = (isPresetAgree and (presetMatIdSel!=0))
			local isVolumeAgree = (volumeVal != undefined)

			local isLocked =
				(not isBasePreset) and  -- UI is never locked for the base preset
				((paramRollout.btn_matLock.checked) or  -- Otherwise, locked if lock was already set, OR ...
				(isForceUpdate))  -- default to locked for presets
			
			local isEditable =
				(not isLocked) and  -- Editing is permitted if unlocked, AND ...
				(isPresetAgree) -- Editing is permitted if all objects agree on a preset (may be "None")
			
			local isMassEditable =
				(isVolumeAgree) or  -- Editing mass if permitted is all objects have same volume, OR ...
				(isBasePreset)  -- All objects set to the "None" preset, so that densities are allowed to vary

			paramRollout.btn_matLock.checked = isLocked
			paramRollout.btn_matLock.visible = isNonBasePreset
				
			paramRollout.spn_density.enabled = isEditable
			paramRollout.spn_mass.enabled = isEditable and isMassEditable
			paramRollout.spn_staticFriction.enabled = isEditable
			paramRollout.spn_dynamicFriction.enabled = isEditable
			paramRollout.spn_bounce.enabled = isEditable
		)
	)

	fn UpdateUI_ParamRolloutValues =
	(
		-- Basic default handling (param rollout)
		-- No special handling needed to set the spinner values.  Yay !!
		if paramRollout!=undefined do
			px_multiedit_methods.UpdateUI_DefaultHandling_RigidBody paramRollout.controlInfoList updateEnables:false
	)
	
	fn UpdateUI_ParamRollout isForceUpdate:false =
	(
		UpdateUI_ParamRolloutValues()
		UpdateUI_ParamRolloutEnables isForceUpdate:isForceUpdate
	)

	fn UpdateUI isForceUpdate:false =
	(
		-- Basic default handling (master rollout)
		px_multiedit_methods.UpdateUI_DefaultHandling_RigidBody controlInfoList

		presetMatIdSel = (px_multiedit_methods.GetParamFromRigidBodyList #materialId)

		local isPresetSelected = ( (presetMatIdSel != undefined) and (presetMatIdSel != 0) )

		btn_deletePreset.enabled = isPresetSelected

		UpdateUI_PresetMatList isForceUpdate:isForceUpdate
		UpdateUI_ParamRollout isForceUpdate:isForceUpdate
	)


	fn SetParamRollout rolloutVal =
	(
		paramRollout = rolloutVal
	)

	fn SetParamToPreset presetMatId paramIdent paramVal =
	(
		case paramIdent of
		(
		#name:				nvpx.materialSetParam presetMatId "Name" (paramVal as string)
		#density:			nvpx.materialSetParam presetMatId "Density" (paramVal as string)
		#staticFriction:	nvpx.materialSetParam presetMatId "StaticFriction" (paramVal as string)
		#dynamicFriction:	nvpx.materialSetParam presetMatId "Friction" (paramVal as string)
		#bounciness:		nvpx.materialSetParam presetMatId "Bounciness" (paramVal as string)
		#mass:
			(
				-- Special case, get the volume if all objects agree, to define density from the mass
				-- TO DO: Optimize this if necessary, may be slow
				local volumeVal = px_multiedit_methods.GetParamFromRigidBodyList #mass
				if volumeVal != undefined and volumeVal>0 do
					nvpx.materialSetParam presetMatId "Density" (((paramVal as float) / (volumeVal as float)) as string)
			)
		)
	)

	fn SetParamToPresetOrRigidBodyList paramIdent paramVal =
	(
		local isPresetSelected = ( (presetMatIdSel != undefined) and (presetMatIdSel != 0) )

		if isPresetSelected then
			SetParamToPreset presetMatIdSel paramIdent paramVal
		else
			px_multiedit_methods.SetParamToRigidBodyList paramIdent paramVal

		-- TO DO: Optimize this if necessary, may be slow
		UpdateUI_ParamRollout() -- Changing mass or density affects the other, update the UI
	)

	fn OnMatLockChanged val =
	(
		UpdateUI_ParamRolloutEnables()
	)

	on ddl_presetMatList selected val do
	(
		-- TO DO: Define controlValMapping so this can use the default handling
		
		local presetMatId = presetMatIdList[ val ] 
		
		if presetMatId==undefined then
			ddl_presetMatList.selection = presetMatIndexSel  -- Reject attempt to pick formatting entries in the dropdown, like "--------"
		else
		(
			-- Apply the material to selected objects
			--px_multiedit_methods.SetParamToRigidBodyList #materialId presetMatId
			for modVal in pxSelectedRigidBodyMods do modVal.SetRBPresetMaterial presetMatId
						
			-- Update UI
			UpdateUI isForceUpdate:true
		)
	)

	on btn_pickMat changed val do
	(
		if val==true do
		(
			-- Pick a node
			local nodeVal = pickObject filter:PxIsRigidBody
			btn_pickMat.checked = false

			-- Get the rigid body
			local modVal = PxGetModRB nodeVal
			if modVal != undefined do
			(
				presetMatIdSel = modVal.materialID -- Get the material from the object
			
				-- Apply the material to selected objects
				px_multiedit_methods.SetParamToRigidBodyList #materialId presetMatIdSel
			
				-- Update UI
				UpdateUI isForceUpdate:true			
			)
		)
	)
	
	on btn_createPreset pressed do
	(
		-- Prompt for a name
		local nameVal = PromptForNameHandler.PromptForName nvpxText.panelMultiMatPromptName nvpxText.panelMultiMatUnnamed
		
		if nameVal!=undefined do
		(
			-- Create the material
			local idVal = nvpx.materialCreate 1
			SetParamToPreset idVal #name nameVal
			SetParamToPreset idVal #density paramRollout.spn_density.value
			SetParamToPreset idVal #staticFriction paramRollout.spn_staticFriction.value
			SetParamToPreset idVal #dynamicFriction paramRollout.spn_dynamicFriction.value
			SetParamToPreset idVal #bounciness paramRollout.spn_bounce.value

			-- Assign new material to selected Rigid Body objects
			--px_multiedit_methods.SetParamToRigidBodyList #materialId idVal
			for modVal in pxSelectedRigidBodyMods do modVal.SetRBPresetMaterial idVal
			
			-- Update UI
			UpdateUI isForceUpdate:true
		)
	)

	on btn_deletePreset pressed do
	(
		-- Remove the material internally
		nvpx.materialRemove presetMatIdSel
		
		-- Assign "Custom" material mode to selected Rigid Body objects
		-- px_multiedit_methods.SetParamToRigidBodyList #materialId 0
		for modVal in pxSelectedRigidBodyMods do modVal.SetRBPresetMaterial 0

		-- Create the array of name with the current selection removed
		local itemsArray = ddl_presetMatList.items
		local itemIndex = ddl_presetMatList.selection
		deleteItem itemsArray itemIndex

		-- Apply the array of names to the dropdown list
		ddl_presetMatList.items = itemsArray

		UpdateUI isForceUpdate:true
	)
	
	on px_panel_multiedit_materialList open do
	(
		px_panel_multiedit_materialList.title = nvpxText.panelMultiMat
		lbl_presetMatList.text = nvpxText.panelMultiMatPreset
		btn_createPreset.text = nvpxText.panelMultiMatPresetCreate
		btn_deletePreset.text = nvpxText.panelMultiMatPresetDelete
	)

	on px_panel_multiedit_materialList help do (HelpSystem.ShowProductHelp 15012) --id_massfx_tools_edit
)


---------- ---------- ---------- ----------
-- Material Properties Rollout
-- Operates as subsidiary of Material List Rollout, forwarding certain calls
---------- ---------- ---------- ----------

rollout px_panel_multiedit_material "Physical Material Properties" rolledUp:false category:4
(
	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC

	checkbutton btn_matLock "L" pos:[8,8] width:24 height:22 checked:true iconName:"Common/Lock" iconSize:[16,16]
	button btn_densityMassLink "" pos:[44,9] width:16 height:31 iconName:"MassFX/DensityMassLink" iconSize:[16,30] border:false visible:true enabled:false
	spinner spn_density "Density   " pos:[69,8] width:104 height:16 type:#float range:[0.01,1000.0,1.0]
	spinner spn_mass "Mass   " pos:[80,30] width:93 height:16 type:#float range:[0.001,PxMaxValue,1.0]
	spinner spn_staticFriction "Static Friction   " pos:[44,52] width:129 height:16 type:#float range:[0.0,1.0,1.0] scale:0.01
	spinner spn_dynamicFriction "Dynamic Friction   " pos:[33,74] width:140 height:16 type:#float range:[0.0,1.0,1.0] scale:0.01
	spinner spn_bounce "Bounciness   " pos:[55,96] width:118 height:16 type:#float range:[0.0,1.0,1.0] scale:0.01

	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC

	local controlInfoList =
	#(
		(MultiEdit_ControlInfo_struct		controlItem:btn_matLock),
		(MultiEdit_ControlInfo_struct		controlItem:spn_density				paramIdent:#density),
		(MultiEdit_ControlInfo_struct		controlItem:spn_mass				paramIdent:#mass),
		(MultiEdit_ControlInfo_struct		controlItem:spn_staticFriction		paramIdent:#staticFriction),
		(MultiEdit_ControlInfo_struct		controlItem:spn_dynamicFriction		paramIdent:#dynamicFriction),
		(MultiEdit_ControlInfo_struct		controlItem:spn_bounce				paramIdent:#bounciness)
	)

	local masterRollout = px_panel_multiedit_materialList

	on btn_matLock changed state do			masterRollout.OnMatLockChanged val			-- Forward call to master rollout
	on spn_density changed val do			masterRollout.SetParamToPresetOrRigidBodyList #density val			-- Forward call to master rollout
	on spn_mass changed val do				masterRollout.SetParamToPresetOrRigidBodyList #mass val				-- Forward call to master rollout
	on spn_staticFriction changed val do	masterRollout.SetParamToPresetOrRigidBodyList #staticFriction val	-- Forward call to master rollout
	on spn_dynamicFriction changed val do	masterRollout.SetParamToPresetOrRigidBodyList #dynamicFriction val	-- Forward call to master rollout
	on spn_bounce changed val do			masterRollout.SetParamToPresetOrRigidBodyList #bounciness val		-- Forward call to master rollout

	fn IsVisible =
	(
		(px_multiedit_methods.IsTabVisible()) and
			px_multiedit_methods.showRigidBodyRollouts
	)

	fn UpdateUI =
	(
		masterRollout.UpdateUI_ParamRollout() -- Forward call to master rollout
	)

	on px_panel_multiedit_material open do
	(
		masterRollout.SetParamRollout px_panel_multiedit_material
		
		px_panel_multiedit_material.title = nvpxText.panelMultiMatProp
		spn_density.text = nvpxText.panelMultiMatPropDensity
		spn_mass.text =  nvpxText.panelMultiMatPropMass
		spn_staticFriction.text =  nvpxText.panelMultiMatPropStaticFriction
		spn_dynamicFriction.text =  nvpxText.panelMultiMatPropDynFriction
		spn_bounce.text =  nvpxText.panelMultiMatPropBounce
	)

	on px_panel_multiedit_material help do (HelpSystem.ShowProductHelp 15012) --id_massfx_tools_edit
)


---------- ---------- ---------- ----------
-- Physical Mesh Rollout
---------- ---------- ---------- ----------

rollout px_panel_multiedit_mesh "Physical Mesh" rolledUp:false category:5
(
	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC

	label lbl_meshType "Mesh Type" pos:[8,11] width:60 height:16
	dropdownList ddl_meshType "" pos:[74,8] width:100 height:21 -- items assigned at runtime

	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC

	struct MeshTypeLookup_struct
	(	-- internal order is Sphere, Box, Capsule, Convex, Original, Custom, Composite
		-- display order is  Sphere, Box, Capsule, Convex, Composite, Original (only for static), Custom
		mode = 1, -- StaticRB is 1, DynamicRB is 2
		controlToParam = #( #(1,2,3,4,7,5,6), #(1,2,3,4,7,6) ),		-- two lookups, depending on mode
		paramToControl = #( #(1,2,3,4,6,7,5), #(1,2,3,4,4,6,5) ),	-- two lookups, depending on mode
		fn SetMode isStaticRB = (mode = (if isStaticRB then 1 else 2)),
		fn ControlToParamFn controlItem controlVal = ( if controlVal==undefined then undefined else  controlToParam[mode][controlVal] ),
		fn ParamToControlFn controlItem paramVal =   ( if paramVal==undefined then undefined else    paramToControl[mode][paramVal] )
	)
	local meshTypeLookup = MeshTypeLookup_struct()

	local controlInfoList =
	#(
		(MultiEdit_ControlInfo_struct		controlItem:lbl_meshType),
		(MultiEdit_ControlInfo_struct		controlItem:ddl_meshType			paramIdent:#MeshType  controlValLookup:meshTypeLookup )
	)

	local paramRollouts = #()
	local meshTypeCur = #uninitialized  -- TO DO: Maybe cache value like this should be owned by px_multiedit_methods
	local meshTypeNames = undefined	
	local meshTypeNames_StaticRB =
			#( nvpxText.panelMultiPhyMeshTSphere, nvpxText.panelMultiPhyMeshTBox, nvpxText.panelMultiPhyMeshTCapsule,
			   nvpxText.panelMultiPhyMeshTConvex, nvpxText.panelMultiPhyMeshTComposite,
			   nvpxText.panelMultiPhyMeshTOriginal, nvpxText.panelMultiPhyMeshTCustom )
	local meshTypeNames_DynamicRB =
			#( nvpxText.panelMultiPhyMeshTSphere, nvpxText.panelMultiPhyMeshTBox, nvpxText.panelMultiPhyMeshTCapsule,
			   nvpxText.panelMultiPhyMeshTConvex, nvpxText.panelMultiPhyMeshTComposite,
			   nvpxText.panelMultiPhyMeshTCustom )
	local meshTypeNames_MultiMeshError = #( "(Multiple Meshes)" ) -- TO DO: Localize this

	fn IsVisible =
	(
		(px_multiedit_methods.IsTabVisible()) and
			px_multiedit_methods.showRigidBodyRollouts
	)

	fn UpdateUI isForceUpdate:false =
	(
		-- Ensure correct rollouts are displayed
		if isForceUpdate then 
		(
			-- Special case: Multiple physical meshes
			-- If any selected Rigid Body Object has multiple physical meshes, disable the mesh parameters
			local isSinglePhysicalMesh = true
			local isStaticRB = true  -- value may be incorrect (and is not relevant) with multiple physical meshes
			for modVal in pxSelectedRigidBodyMods while isSinglePhysicalMesh do
			(
				if (modVal.getRBMeshCount()) != 1 do isSinglePhysicalMesh = false
				if (modVal.type!=3) do isStaticRB = false
			)
			meshTypeLookup.SetMode isStaticRB  -- update the behavior of the Mesh Type dropdown
			meshTypeNames = (if isStaticRB then meshTypeNames_StaticRB else meshTypeNames_DynamicRB)

			
			-- Need updated meshTypeLookup mode, before the default handling
			-- Basic default handling
			px_multiedit_methods.UpdateUI_DefaultHandling_RigidBody controlInfoList		
			
			
			if isSinglePhysicalMesh then
			(
				-- Current mesh type from the recently update UI, or undefined if selected don't agree
				meshTypeCur = px_multiedit_methods.GetParamFromControlItem ddl_meshType
				ddl_meshType.items = meshTypeNames
				-- ddl_meshType.enabled and .selection are set by the UI update above
			)
			else
			(
				meshTypeCur = undefined
				ddl_meshType.enabled = false
				ddl_meshType.selection = 1
				ddl_meshType.items = meshTypeNames_MultiMeshError
			)

			-- Display the correct parameter rollout, or none in the special case
			px_multiedit_methods.UpdateUI_TabRollouts()
		)
		else
		(
			-- Basic default handling
			px_multiedit_methods.UpdateUI_DefaultHandling_RigidBody controlInfoList		
		)
	)

	fn IsMeshTypeVisible meshTypeVal =
	(
		-- Hack: this might be called after rollout is created, but before first draw.  Special case initialization
		if meshTypeCur == #uninitialized do meshTypeCur = (px_multiedit_methods.GetParamFromRigidBodyList #MeshType)
		meshTypeVal == meshTypeCur
	)

	fn SetParamRollout meshTypeVal rolloutVal =
	(
		paramRollouts[meshTypeVal] = rolloutVal
	)
		

	on ddl_meshType selected val do
	(
		px_multiedit_methods.SetControlToRigidBodyList ddl_meshType val
		UpdateUI isForceUpdate:true
		
		if meshTypeCur!=undefined do paramRollouts[meshTypeCur].UpdateUI isForceUpdate:true
	)

	fn OnPickCustomMesh pickedNode =
	(
		px_multiedit_methods.SetParamToRigidBodyList #meshCustomMesh pickedNode
	)
	fn OnSelectCustomMesh =
	(
		local nodesToSelect = #()
		for modVal in pxSelectedRigidBodyMods do
		(
			if modVal.meshCustomMesh != undefined do ( append nodesToSelect modVal.meshCustomMesh )
		)
		if nodesToSelect.count > 0 do select nodesToSelect
	)
	fn OnExtractCustomMesh =
	(
		for modVal in pxSelectedRigidBodyMods do
		(
			if modVal.meshCustomMesh == undefined do modVal.ExtractCustomMesh()
		)
	)
	fn OnUpdateCustomMesh =
	(
		for modVal in pxSelectedRigidBodyMods do
		(
			modVal.meshCustomMesh = modVal.meshCustomMesh
		)
	)
	fn OnGenerateCompositeMesh =
	(
		for modVal in pxSelectedRigidBodyMods do modVal.RBMeshCompositeGenerate 1
	)

	fn OnCompositeMeshAdvancedDialog =
	(
		for modVal in pxSelectedRigidBodyMods do modVal.RBMeshCompositeAdvancedDialog()
	)

	on px_panel_multiedit_mesh open do
	(
		ddl_meshType.items = makeUniqueArray meshTypeNames_StaticRB -- TO DO: Is this needed?
		
		px_panel_multiedit_mesh.title = nvpxText.panelMultiPhyMesh
		lbl_meshType.text = nvpxText.panelMultiPhyMeshType
	)

	on px_panel_multiedit_mesh help do (HelpSystem.ShowProductHelp 15012) --id_massfx_tools_edit
)


---------- ---------- ---------- ----------
-- Sphere Parameters Rollout
-- Operates as subsidiary of Physical Mesh Rollout, forwarding certain calls
---------- ---------- ---------- ----------

rollout px_panel_multiedit_sphereParams "Physical Mesh Parameters" category:6
(
	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC

	spinner spn_sphRadius "Radius   " pos:[62,8] width:112 height:16 type:#float range:[0.0,PxMaxValue,0.0]

	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC

	local controlInfoList =
	#(
		(MultiEdit_ControlInfo_struct		controlItem:spn_sphRadius			paramIdent:#meshRadius)
	)

	local masterRollout = px_panel_multiedit_mesh
	local meshType = PX_MESHTYPE_SPHERE

	on spn_sphRadius changed val do			px_multiedit_methods.SetControlToRigidBodyList spn_sphRadius val

	fn IsVisible =
	(
		(px_multiedit_methods.IsTabVisible()) and
			(px_multiedit_methods.showRigidBodyRollouts) and (masterRollout.IsMeshTypeVisible meshType)
	)

	fn UpdateUI =
	(
		px_multiedit_methods.UpdateUI_DefaultHandling_RigidBody controlInfoList
	)

	on px_panel_multiedit_sphereParams open do
	(
		masterRollout.SetParamRollout meshType px_panel_multiedit_sphereParams
		
		px_panel_multiedit_sphereParams.title = nvpxText.panelMultiPhyMeshParam
		spn_sphRadius.text = nvpxText.panelMultiPhyMeshParamRadius
	)

	on px_panel_multiedit_sphereParams help do (HelpSystem.ShowProductHelp 15012) --id_massfx_tools_edit
)


---------- ---------- ---------- ----------
-- Box Parameters Rollout
-- Operates as subsidiary of Physical Mesh Rollout, forwarding certain calls
---------- ---------- ---------- ----------

rollout px_panel_multiedit_boxParams "Physical Mesh Parameters" category:5
(
	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC

	spinner spn_boxLength "Length   " pos:[62,8] width:112 height:16 type:#float range:[0.0,PxMaxValue,0.0]
	spinner spn_boxWidth "Width   " pos:[67,30] width:107 height:16 type:#float range:[0.0,PxMaxValue,0.0]
	spinner spn_boxHeight "Height   " pos:[64,52] width:110 height:16 type:#float range:[0.0,PxMaxValue,0.0]

	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC

	local controlInfoList =
	#(
		(MultiEdit_ControlInfo_struct		controlItem:spn_boxLength			paramIdent:#meshLength),
		(MultiEdit_ControlInfo_struct		controlItem:spn_boxWidth			paramIdent:#meshWidth),
		(MultiEdit_ControlInfo_struct		controlItem:spn_boxHeight			paramIdent:#meshHeight)
	)

	local masterRollout = px_panel_multiedit_mesh
	local meshType = PX_MESHTYPE_BOX

	on spn_boxLength changed val do			px_multiedit_methods.SetControlToRigidBodyList spn_boxLength val
	on spn_boxWidth changed val do			px_multiedit_methods.SetControlToRigidBodyList spn_boxWidth val
	on spn_boxHeight changed val do			px_multiedit_methods.SetControlToRigidBodyList spn_boxHeight val

	fn IsVisible =
	(
		(px_multiedit_methods.IsTabVisible()) and
			(px_multiedit_methods.showRigidBodyRollouts) and (masterRollout.IsMeshTypeVisible meshType)
	)

	fn UpdateUI =
	(
		px_multiedit_methods.UpdateUI_DefaultHandling_RigidBody controlInfoList
	)

	on px_panel_multiedit_boxParams open do
	(
		masterRollout.SetParamRollout meshType px_panel_multiedit_boxParams
		
		px_panel_multiedit_boxParams.title = nvpxText.panelMultiPhyMeshParam
		spn_boxLength.text = nvpxText.panelMultiPhyMeshParamLength
		spn_boxWidth.text = nvpxText.panelMultiPhyMeshParamWidth
		spn_boxHeight.text = nvpxText.panelMultiPhyMeshParamHeight
	)

	on px_panel_multiedit_boxParams help do (HelpSystem.ShowProductHelp 15012) --id_massfx_tools_edit
)


---------- ---------- ---------- ----------
-- Capsule Parameters Rollout
-- Operates as subsidiary of Physical Mesh Rollout, forwarding certain calls
---------- ---------- ---------- ----------

rollout px_panel_multiedit_capsuleParams "Physical Mesh Parameters" category:6
(
	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC

	spinner spn_capsuleRad "Radius   " pos:[63,8] width:111 height:16 type:#float range:[0.0,PxMaxValue,0.0]
	spinner spn_capsuleHeight "Height   " pos:[64,30] width:110 height:16 type:#float range:[0.0,PxMaxValue,0.0]

	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC

	local controlInfoList =
	#(
		(MultiEdit_ControlInfo_struct		controlItem:spn_capsuleRad			paramIdent:#MeshRadius),
		(MultiEdit_ControlInfo_struct		controlItem:spn_capsuleHeight		paramIdent:#MeshHeight)
	)

	local masterRollout = px_panel_multiedit_mesh
	local meshType = PX_MESHTYPE_CAPSULE

	on spn_capsuleRad changed val do		px_multiedit_methods.SetControlToRigidBodyList spn_capsuleRad val
	on spn_capsuleHeight changed val do		px_multiedit_methods.SetControlToRigidBodyList spn_capsuleHeight val

	fn IsVisible =
	(
		(px_multiedit_methods.IsTabVisible()) and
			(px_multiedit_methods.showRigidBodyRollouts) and (masterRollout.IsMeshTypeVisible meshType)
	)

	fn UpdateUI =
	(
		px_multiedit_methods.UpdateUI_DefaultHandling_RigidBody controlInfoList
	)

	on px_panel_multiedit_capsuleParams open do
	(
		masterRollout.SetParamRollout meshType px_panel_multiedit_capsuleParams
		
		px_panel_multiedit_capsuleParams.title = nvpxText.panelMultiPhyMeshParam
		spn_capsuleRad.text = nvpxText.panelMultiPhyMeshParamRadius
		spn_capsuleHeight.text = nvpxText.panelMultiPhyMeshParamHeight
	)

	on px_panel_multiedit_capsuleParams help do (HelpSystem.ShowProductHelp 15012) --id_massfx_tools_edit
)


---------- ---------- ---------- ----------
-- Convex Parameters Rollout
-- Operates as subsidiary of Physical Mesh Rollout, forwarding certain calls
---------- ---------- ---------- ----------

rollout px_panel_multiedit_convexParams "Physical Mesh Parameters" category:6
(
	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC
	
	label lbl_convexVertCount "-- Vertices in Mesh" align:#center offset:[0,4]
	spinner spn_inflation "Inflation" across:1 offset:[0,6] type:#float range:[-PxMaxValue,PxMaxValue,0.0]
	label lbl_convexStyle "Generate From" across:2 offset:[0,4]
	dropdownList ddl_convexStyle "" across:1 items:#("Surface", "Vertices")
	spinner spn_verticesLimit "Vertices" across:1 type:#integer range:[0,PxMaxValue,0]
	button btn_generateNow "Regenerate" across:1 align:#center offset:[0,4]
	
	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC

	struct ConvexStyleLookup_struct
	(
		fn ControlToParamFn controlItem controlVal = ( if controlVal==undefined then undefined else (controlVal-1) ),
		fn ParamToControlFn controlItem paramVal =   ( if paramVal==undefined then undefined else (paramVal+1) )
	)
	local convexStyleLookup = ConvexStyleLookup_struct()

	local controlInfoList =
	#(
		(MultiEdit_ControlInfo_struct		controlItem:lbl_convexVertCount),
		(MultiEdit_ControlInfo_struct		controlItem:spn_inflation			paramIdent:#meshInflation),
		(MultiEdit_ControlInfo_struct		controlItem:lbl_convexStyle),
		(MultiEdit_ControlInfo_struct		controlItem:ddl_convexStyle			paramIdent:#meshConvexStyle controlValLookup:convexStyleLookup),
		(MultiEdit_ControlInfo_struct		controlItem:spn_verticesLimit		paramIdent:#meshVerticesLimit)
	)

	local masterRollout = px_panel_multiedit_mesh
	local meshType = PX_MESHTYPE_CONVEX

	fn GenerateNowFunc itemVal argVal = ( itemVal.RBMeshRegenerate 1 )

	on spn_Inflation changed val do			px_multiedit_methods.SetControlToRigidBodyList spn_Inflation val
	on ddl_convexStyle selected val do		px_multiedit_methods.SetControlToRigidBodyList ddl_convexStyle val
	on spn_verticesLimit changed val do		px_multiedit_methods.SetControlToRigidBodyList spn_verticesLimit val
	on btn_generateNow pressed do			px_multiedit_methods.InvokeFuncOnRigidBodyList GenerateNowFunc undefined

	fn GetMeshVertexCountFunc itemVal argVal = 
	(
		local mesh = itemVal.GetRBMeshShape 1  --assume exactly one physical mesh
		if mesh!=undefined then mesh.numVerts else undefined
	)
	
	fn UpdateUI_Label =
	(
		local vertexCount =
			px_multiedit_methods.GetFuncResultFromRigidBodyList GetMeshVertexCountFunc undefined

		lbl_convexVertCount.text =
			if vertexCount == undefined then
				nvpxText.panelMultiPhyMeshParamVertCountUnmatch
			else
            (
				local strStream = StringStream("")
				format nvpxText.panelMultiPhyMeshParamVertCount vertexCount to:strStream
				strStream as string
            )
	)
	
	fn IsVisible =
	(
		(px_multiedit_methods.IsTabVisible()) and
			(px_multiedit_methods.showRigidBodyRollouts) and (masterRollout.IsMeshTypeVisible meshType)
	)

	fn UpdateUI =
	(
		px_multiedit_methods.UpdateUI_DefaultHandling_RigidBody controlInfoList

		UpdateUI_Label()
	)

	on px_panel_multiedit_convexParams open do
	(
		masterRollout.SetParamRollout meshType px_panel_multiedit_convexParams
		
		px_panel_multiedit_convexParams.title = nvpxText.panelMultiPhyMeshParam
		lbl_convexVertCount.text = nvpxText.panelMultiPhyMeshParamVertCount
		spn_inflation.text = nvpxText.panelMultiPhyMeshParamInflation
		lbl_convexStyle.text = nvpxText.panelMultiPhyMeshParamGenerateFrom
		spn_verticesLimit.text = nvpxText.panelMultiPhyMeshParamVerts
		btn_generateNow.text = nvpxText.panelMultiPhyMeshParamGenerateNow
		ddl_convexStyle.items = #( nvpxText.panelMultiConvexParamSurface, nvpxText.panelMultiConvexParamVertices )
	)

	on px_panel_multiedit_convexParams help do (HelpSystem.ShowProductHelp 15012) --id_massfx_tools_edit
)


---------- ---------- ---------- ----------
-- Composite Parameters Rollout
-- Operates as subsidiary of Physical Mesh Rollout, forwarding certain calls
---------- ---------- ---------- ----------

rollout px_panel_multiedit_compositeParams "Physical Mesh Parameters" category:6
(
	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC

	-- TO DO: Localize this
	label lbl_compositeMaxConcavity "Mesh Detail" align:#left across:3 offset:[0,6]
	spinner spn_compositeMaxConcavity "" width:50 align:#left offset:[40,6] type:#float range:[0.0,100.0,0.0]
	label lbl_compositeMaxConcavityPercent "%" align:#left offset:[43,6]
	
	GroupBox grp_advanced "Advanced Parameters" width:165 height:15 align:#center across:1 offset:[0,6]
	label lbl_smallClusterThreshold "Min. Hull Size" align:#left across:3 offset:[2,3]
	spinner spn_smallClusterThreshold "" width:50 align:#left offset:[39,3] type:#float range:[0.0,100.0,0.0]
	label lbl_smallClusterThresholdPercent "%" align:#left offset:[43,3]
	label lbl_compositeMaxVertices "Max. Verts Per Hull" width:60 height:28 align:#left across:2 offset:[2,0]
	spinner spn_compositeMaxVertices "" width:50 align:#left offset:[14,0] type:#integer range:[4,256,1]
	checkbox chk_compositeHighQuality "Improve Fitting" align:#left offset:[0,3]
	
	button btn_generateCompositeMesh width:166 offset:[0,16]
	
	label lbl_hulls_verts "Hulls / Verts" align:#center offset:[0,6]
	
	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC

	local controlInfoList =
	#(
		(MultiEdit_ControlInfo_struct		controlItem:spn_compositeMaxConcavity	paramIdent:#compositeMaxConcavity),
		(MultiEdit_ControlInfo_struct		controlItem:spn_smallClusterThreshold		paramIdent:#smallClusterThreshold),
		(MultiEdit_ControlInfo_struct		controlItem:spn_compositeMaxVertices	paramIdent:#compositeMaxVertices),
		(MultiEdit_ControlInfo_struct		controlItem:chk_compositeHighQuality		paramIdent:#compositeHighQuality)
	)

	local masterRollout = px_panel_multiedit_mesh
	local meshType = PX_MESHTYPE_COMPOSITE

	fn GetCompositeHullCountFunc itemVal argVal = 
	(
		local idx = itemVal.GetRBMeshCurrentIndex()
		if idx >= 0 then
		(
			local hullCount = itemVal.GetRBMeshCompositeMeshCount idx
			-- TO DO: Was the GetRBMeshCompositeMeshVerts() function removed? MaxScript break?

			hullCount  -- return value
		)	
	)

	fn UpdateUI_Labels =
	(
		local hullCount =
			px_multiedit_methods.GetFuncResultFromRigidBodyList GetCompositeHullCountFunc undefined

		-- Hull and vertex count is a Point2, hullCount is x, vertexCount is y
		lbl_hulls_verts.text =
			if hullCount == undefined then
				nvpxText.panelMultiPhyMeshParamHullsUnmatch
			else
			(
				local hullCount = (hullCount as integer)
				local hullCountStr = (if hullCount==0 then "--" else (hullCount as string))
				local strStream = StringStream("")
				format nvpxText.panelMultiPhyMeshParamHullsFormat hullCountStr to:strStream
				strStream as string
			)
	)

	on spn_compositeMaxConcavity changed val do	px_multiedit_methods.SetControlToRigidBodyList spn_compositeMaxConcavity val
	on spn_smallClusterThreshold changed val do		px_multiedit_methods.SetControlToRigidBodyList spn_smallClusterThreshold val
	on spn_compositeMaxVertices changed val do		px_multiedit_methods.SetControlToRigidBodyList spn_compositeMaxVertices val
	on chk_compositeHighQuality changed val do		px_multiedit_methods.SetControlToRigidBodyList chk_compositeHighQuality val
	on btn_generateCompositeMesh pressed do
	(
		lbl_hulls_verts.text = nvpxText.panelMultiPhyMeshParamCalculating
		masterRollout.OnGenerateCompositeMesh()	-- Forward call to master rollout
		UpdateUI_Labels()
	)
	
	fn IsVisible =
	(
		(px_multiedit_methods.IsTabVisible()) and
			(px_multiedit_methods.showRigidBodyRollouts) and (masterRollout.IsMeshTypeVisible meshType)
	)

	fn UpdateUI =
	(
		px_multiedit_methods.UpdateUI_DefaultHandling_RigidBody controlInfoList

		UpdateUI_Labels()
	)

	on px_panel_multiedit_compositeParams open do
	(
		masterRollout.SetParamRollout meshType px_panel_multiedit_compositeParams

		px_panel_multiedit_compositeParams.title = nvpxText.panelMultiPhyMeshParam		
		lbl_compositeMaxConcavity.text = nvpxText.panelMultiPhyMeshParamCompositeMaxConcavity
		lbl_compositeMaxConcavityPercent.text = nvpxText.panelMultiPhyMeshParamPercent
		grp_advanced.text = nvpxText.panelMultiPhyMeshParamAdvancedGroup
		lbl_smallClusterThreshold.text = nvpxText.panelMultiPhyMeshParamSmallClusterThreshold
		lbl_smallClusterThresholdPercent.text = nvpxText.panelMultiPhyMeshParamPercent
		lbl_compositeMaxVertices.text = nvpxText.panelMultiPhyMeshParamCompositeMaxVertices
		chk_compositeHighQuality.text = nvpxText.panelMultiPhyMeshParamCompositeHighQuality
		chk_compositeHighQuality.tooltip = nvpxText.panelMultiPhyMeshParamCompositeHighQualityTT
		btn_generateCompositeMesh.text = nvpxText.panelMultiPhyMeshParamGenerate
		lbl_hulls_verts.text = ""
		
		grp_advanced.height = 24 + (chk_compositeHighQuality.pos.y - grp_advanced.pos.y)
		px_panel_multiedit_compositeParams.height += 8
	)

	on px_panel_multiedit_compositeParams help do (HelpSystem.ShowProductHelp 15012) --id_massfx_tools_edit
)


---------- ---------- ---------- ----------
-- Original Mesh Parameters Rollout
-- Operates as subsidiary of Physical Mesh Rollout, forwarding certain calls
---------- ---------- ---------- ----------

rollout px_panel_multiedit_originalParams "Physical Mesh Parameters" category:6
(
	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC
	
	label lbl_noParams "This mesh type has no parameters" pos:[6,8] width:168 height:16

	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC
	
	local controlInfoList =
	#(
		(MultiEdit_ControlInfo_struct		controlItem:lbl_noParams)
	)

	local masterRollout = px_panel_multiedit_mesh
	local meshType = PX_MESHTYPE_ORIGINAL

	fn IsVisible =
	(
		(px_multiedit_methods.IsTabVisible()) and
			(px_multiedit_methods.showRigidBodyRollouts) and (masterRollout.IsMeshTypeVisible meshType)
	)

	fn UpdateUI =
	(
		px_multiedit_methods.UpdateUI_DefaultHandling_RigidBody controlInfoList
	)

	on px_panel_multiedit_originalParams open do
	(
		masterRollout.SetParamRollout meshType px_panel_multiedit_originalParams
		
		px_panel_multiedit_originalParams.title = nvpxText.panelMultiPhyMeshParam
		lbl_noParams.text = nvpxText.panelMultiPhyMeshParamNoParam
	)

	on px_panel_multiedit_originalParams help do (HelpSystem.ShowProductHelp 15012) --id_massfx_tools_edit
)


---------- ---------- ---------- ----------
-- Custom Mesh Parameters Rollout
-- Operates as subsidiary of Physical Mesh Rollout, forwarding certain calls
---------- ---------- ---------- ----------

rollout px_panel_multiedit_customParams "Physical Mesh Parameters" category:6
(
	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC
	
	pickbutton btn_pickCustomMesh "Pick Source Object" align:1 width:150
	button btn_selectCustomMesh "Select Source Object" align:1 width:150
	button btn_extractCustomMesh "Extract To Source Object" align:1 width:150
	button btn_updateCustomMesh "Update From Source Object" align:1 width:150

	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC
	
	local controlInfoList =
	#(
		(MultiEdit_ControlInfo_struct		controlItem:btn_pickCustomMesh),
		(MultiEdit_ControlInfo_struct		controlItem:btn_selectCustomMesh),
		(MultiEdit_ControlInfo_struct		controlItem:btn_extractCustomMesh),
		(MultiEdit_ControlInfo_struct		controlItem:btn_updateCustomMesh)
	)

	local masterRollout = px_panel_multiedit_mesh
	local meshType = PX_MESHTYPE_CUSTOM

	fn UpdateUI_Pick =
	(
		local textPick = nvpxText.panelMultiPhyMeshParamPickSource
		local customMeshVal = px_multiedit_methods.GetParamFromRigidBodyList #meshCustomMesh

		if customMeshVal != undefined do textPick = customMeshVal.name

		btn_pickCustomMesh.text = textPick 

		local bakedVal = px_multiedit_methods.GetParamFromRigidBodyList #baked
		if bakedVal!=undefined do bakedVal = (bakedVal>0)  -- Bake modes other than zero indicate baked object

		local oneMeshDefined = false
		local oneMeshUndefined = false

		for mod in pxSelectedRigidBodyMods do
		(
			if mod.meshCustomMesh == undefined then ( oneMeshUndefined = true )
			else ( oneMeshDefined = true )
		)
		
		-- Enable controls if none of the selected objects are baked
		btn_pickCustomMesh.enabled = (bakedVal==false)
		btn_selectCustomMesh.enabled = (bakedVal==false)
		btn_extractCustomMesh.enabled = (bakedVal==false) and (oneMeshUndefined)
		btn_updateCustomMesh.enabled = (bakedVal==false) and (oneMeshDefined)
	)

	on btn_pickCustomMesh picked obj do
	(
		masterRollout.OnPickCustomMesh( obj )	-- Forward call to master rollout
		UpdateUI_Pick()
	)
	on btn_selectCustomMesh pressed do	masterRollout.OnSelectCustomMesh()	-- Forward call to master rollout

	on btn_extractCustomMesh pressed do
	(
		masterRollout.OnExtractCustomMesh()	-- Forward call to master rollout
		UpdateUI_Pick()
	)

	on btn_updateCustomMesh pressed do	masterRollout.OnUpdateCustomMesh()	-- Forward call to master rollout

	fn IsVisible =
	(
		(px_multiedit_methods.IsTabVisible()) and
			(px_multiedit_methods.showRigidBodyRollouts) and (masterRollout.IsMeshTypeVisible meshType)
	)

	fn UpdateUI =
	(
		px_multiedit_methods.UpdateUI_DefaultHandling_RigidBody controlInfoList

		UpdateUI_Pick()
	)

	on px_panel_multiedit_customParams open do
	(
		masterRollout.SetParamRollout meshType px_panel_multiedit_customParams
		
		px_panel_multiedit_customParams.title = nvpxText.panelMultiPhyMeshParam
		btn_pickCustomMesh.text = nvpxText.panelMultiPhyMeshParamPickSource
		btn_selectCustomMesh.text = nvpxText.panelMultiPhyMeshParamSelectSource
		btn_extractCustomMesh.text = nvpxText.panelMultiPhyMeshParamExtractToSource
		btn_updateCustomMesh.text = nvpxText.panelMultiPhyMeshParamUpdateFromSource
	)

	on px_panel_multiedit_customParams help do (HelpSystem.ShowProductHelp 15012) --id_massfx_tools_edit
)


---------- ---------- ---------- ----------
-- Advanced Rollout
---------- ---------- ---------- ----------

rollout px_panel_multiedit_advanced "Advanced" rolledUp:true category:7
(
	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC
	
	GroupBox grp_simulationOverrides "Simulation" width:165 height:15 align:#center across:1
	checkbox chk_overSolverIter "Override Solver Iterations" align:#left across:1 offset:[0,3]
	label lbl_overSolverIterVal "" across:2
	spinner spn_overSolverIterVal "" width:60 align:#right type:#integer range:[0,512,1]
	checkbox chk_enableBackfaceCollision  "Enable Backface Collision" align:#left

	GroupBox grp_contactShell "Contact Shell" width:165 height:15 align:#center across:1 offset:[0,11]
	checkbox chk_contact_overrideGlobals "Override Globals" align:#left offset:[0,3]
	label lbl_contact_distance "Contact Distance" align:#right across:2 offset:[9,3]
	spinner spn_contact_distance "" width:60 align:#right offset:[0,3] type:#float range:[-1000,1000,1]
	label lbl_contact_restDepth "Rest Depth" align:#right across:2 offset:[9,0]
	spinner spn_contact_restDepth "" width:60 align:#right type:#float range:[-1000,1000,1]

	GroupBox grp_initialMotion "Initial Motion" width:165 height:15 align:#center across:1 offset:[0,10]
	
	radiobuttons rdo_advMode "" labels:#(
		nvpxText.panelMultiAdvancedAbs,
		nvpxText.panelMultiAdvancedRel ) \
		columns:2 offset:[0,5]
	label lbl_initialVelocity "Initial Velocity" align:#left across:3 offset:[2,8]
	label lbl_velX "X" align:#right offset:[-16,8]
	spinner spn_velX "" width:60 align:#right offset:[0,8] type:#float range:[-PxMaxValue,PxMaxValue,0.0]
	label lbl_velY "Y" align:#right across:2 offset:[9,0]
	spinner spn_velY "" width:60 align:#right type:#float range:[-PxMaxValue,PxMaxValue,0.0]
	label lbl_velZ "Z" align:#right across:2 offset:[9,1]
	spinner spn_velZ "" width:60 align:#right type:#float range:[-PxMaxValue,PxMaxValue,0.0]
	label lbl_velSpeed "Speed" align:#right across:2 offset:[9,5]
	spinner spn_velSpeed "" width:60 align:#right offset:[0,5] type:#float range:[-PxMaxValue,PxMaxValue,0.0]
	label lbl_initialSpin "Initial Spin" align:#left across:3 offset:[2,12]
	label lbl_spinX "X" align:#right offset:[-16,12]
	spinner spn_spinX  "" align:#right width:60 offset:[0,12] type:#float range:[-PxMaxValue,PxMaxValue,0.0]
	label lbl_spinY "Y" align:#right across:2 offset:[9,0]
	spinner spn_spinY "" align:#right width:60 type:#float range:[-PxMaxValue,PxMaxValue,0.0]
	label lbl_spinZ "Z" align:#right across:2 offset:[9,0]
	spinner spn_spinZ "" width:60 align:#right type:#float range:[-PxMaxValue,PxMaxValue,0.0]
	label lbl_spinSpeed "Speed" align:#right across:2 offset:[9,5]
	spinner spn_spinSpeed "" align:#right width:60 offset:[0,5] type:#float range:[-PxMaxValue,PxMaxValue,0.0]

	GroupBox grp_centerOfMass "Center of Mass" width:165 height:15 align:#center across:1 offset:[0,10]
	radiobuttons rdo_massCenterMode "" labels:#(
		nvpxText.panelMultiAdvancedCOMCalculateFromMeshes,
		nvpxText.panelMultiAdvancedCOMUsePivot,
		nvpxText.panelMultiAdvancedCOMLocalOffset ) \
		columns:1 align:#left offsets:#( [0,0], [0,5], [0,10] )
	label lbl_massCenterX "X" align:#right across:2 offset:[9,3]
	spinner spn_massCenterX "" width:60 align:#right offset:[0,3] type:#float range:[-PxMaxValue,PxMaxValue,0.0]
	label lbl_massCenterY "Y" align:#right across:2 offset:[9,0]
	spinner spn_massCenterY "" width:60 align:#right type:#float range:[-PxMaxValue,PxMaxValue,0.0]
	label lbl_massCenterZ "Z" align:#right across:2 offset:[9,0]
	spinner spn_massCenterZ "" width:60 align:#right type:#float range:[-PxMaxValue,PxMaxValue,0.0]

	GroupBox grp_damping "Damping" width:165 height:15 align:#center across:1 offset:[0,10]
	label lbl_linDamping "Linear" align:#left across:2 offset:[2,3]
	spinner spn_linDamping "" width:60 align:#right offset:[0,3] type:#float range:[-PxMaxValue,PxMaxValue,0.0]
	label lbl_angDamping "Angular" align:#left across:2 offset:[2,0]
	spinner spn_angDamping "" width:60 align:#right type:#float range:[-PxMaxValue,PxMaxValue,0.0]

	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC

	struct ZeroToOneBaseLookup_struct
	(
		fn ControlToParamFn controlItem controlVal = ( if controlVal==undefined then undefined else (controlVal-1) ),
		fn ParamToControlFn controlItem paramVal =   ( if paramVal==undefined then undefined else (paramVal+1) )
	)
	local zeroToOneBaseLookup = zeroToOneBaseLookup_struct()

	local controlInfoList =
	#(
		(MultiEdit_ControlInfo_struct		controlItem:grp_simulationOverrides),
		(MultiEdit_ControlInfo_struct		controlItem:chk_overSolverIter				paramIdent:#SolverIter),
		(MultiEdit_ControlInfo_struct		controlItem:spn_overSolverIterVal			paramIdent:#SolverIterValue),
		(MultiEdit_ControlInfo_struct		controlItem:chk_enableBackfaceCollision	paramIdent:#EnableBackfaceCollision),

		(MultiEdit_ControlInfo_struct		controlItem:grp_contactShell),
		(MultiEdit_ControlInfo_struct		controlItem:chk_contact_overrideGlobals	paramIdent:#contactShellOverrideGlobals),
		(MultiEdit_ControlInfo_struct		controlItem:spn_contact_distance			paramIdent:#contactShellContactDistance),
		(MultiEdit_ControlInfo_struct		controlItem:spn_contact_restDepth	    paramIdent:#contactShellRestDepth),
		
		(MultiEdit_ControlInfo_struct		controlItem:grp_initialMotion),
		(MultiEdit_ControlInfo_struct		controlItem:rdo_advMode						paramIdent:#InitialMotionStyle  controlValLookup:zeroToOneBaseLookup),
		(MultiEdit_ControlInfo_struct		controlItem:lbl_initialVelocity),
		(MultiEdit_ControlInfo_struct		controlItem:spn_velX							paramIdent:#InitialVelocityX),
		(MultiEdit_ControlInfo_struct		controlItem:spn_velY							paramIdent:#InitialVelocityY),
		(MultiEdit_ControlInfo_struct		controlItem:spn_velZ							paramIdent:#InitialVelocityZ),
		(MultiEdit_ControlInfo_struct		controlItem:spn_velSpeed						paramIdent:#VelocitySpeed),
		(MultiEdit_ControlInfo_struct		controlItem:lbl_initialSpin),
		(MultiEdit_ControlInfo_struct		controlItem:spn_spinX							paramIdent:#InitialSpinX),
		(MultiEdit_ControlInfo_struct		controlItem:spn_spinY							paramIdent:#InitialSpinY),
		(MultiEdit_ControlInfo_struct		controlItem:spn_spinZ							paramIdent:#InitialSpinZ),
		(MultiEdit_ControlInfo_struct		controlItem:spn_spinSpeed					paramIdent:#SpinSpeed),
		
		(MultiEdit_ControlInfo_struct		controlItem:grp_centerOfMass),
		(MultiEdit_ControlInfo_struct		controlItem:rdo_massCenterMode			paramIdent:#massCenterMode controlValLookup:zeroToOneBaseLookup),
		(MultiEdit_ControlInfo_struct		controlItem:spn_massCenterX				paramIdent:#massCenterX),
		(MultiEdit_ControlInfo_struct		controlItem:spn_massCenterY				paramIdent:#massCenterY),
		(MultiEdit_ControlInfo_struct		controlItem:spn_massCenterZ				paramIdent:#massCenterZ),
		
		(MultiEdit_ControlInfo_struct		controlItem:grp_damping),
		(MultiEdit_ControlInfo_struct		controlItem:spn_linDamping					paramIdent:#LinearDamping),
		(MultiEdit_ControlInfo_struct		controlItem:spn_angDamping					paramIdent:#AngularDamping)
	)

	fn UpdateUI_ContactShellEnables =
	(
		lbl_contact_distance.enabled = lbl_contact_restDepth.enabled =
		spn_contact_distance.enabled = spn_contact_restDepth.enabled =
			chk_contact_overrideGlobals.checked		
	)

	fn UpdateUI_MassCenterEnables =
	(
		lbl_massCenterX.enabled = lbl_massCenterY.enabled = lbl_massCenterZ.enabled =
		spn_massCenterX.enabled = spn_massCenterY.enabled = spn_massCenterZ.enabled =
			(rdo_massCenterMode.state==3)		
	)
	
	fn UpdateUI_StaticRBEnables =
	(
		-- special static RB controls
		local rbType = px_multiedit_methods.GetParamFromRigidBodyList #type

		-- IsDynamicEver means Unbaked, and either Dynamic or Kinematic-to-Dynamic
		local isDynamicEverRB = ( px_multiedit_methods.GetParamFromRigidBodyList #IsDynamicEver)
		local isStaticRB= (rbType==3)
		chk_enableBackfaceCollision.enabled = isStaticRB

		local dynamicOnlyControls =
			#( -- Initial Motion controls
				grp_initialMotion,
					rdo_advMode,
					-- Initial Velocity Controls
					lbl_initialVelocity,
						lbl_velX, lbl_velY, lbl_velZ, lbl_velSpeed, spn_velX, spn_velY, spn_velZ, spn_velSpeed,
					-- Initial Spin controls
					lbl_initialSpin,
						lbl_spinX, lbl_spinY, lbl_spinZ, lbl_spinSpeed, spn_spinX, spn_spinY, spn_spinZ, spn_spinSpeed,
				-- Damping controls
				grp_damping,
					lbl_linDamping, lbl_angDamping, spn_linDamping, spn_angDamping
				)
		local dynamicOrKinematicControls =
			#(	-- Override Solver Iterations controls
				chk_overSolverIter, lbl_overSolverIterVal, spn_overSolverIterVal,
				-- Center of Mass controls
				grp_centerOfMass,
					rdo_massCenterMode,
						lbl_massCenterX, lbl_massCenterY, lbl_massCenterZ, spn_massCenterX, spn_massCenterY, spn_massCenterZ
				)
		for x in dynamicOnlyControls do x.enabled = isDynamicEverRB			
		for x in dynamicOrKinematicControls do x.enabled = (not isStaticRB)
	)
	
	on chk_overSolverIter changed val do					px_multiedit_methods.SetControlToRigidBodyList chk_overSolverIter val
	on spn_overSolverIterVal changed val do				px_multiedit_methods.SetControlToRigidBodyList spn_overSolverIterVal val
	on chk_enableBackfaceCollision changed val do		px_multiedit_methods.SetControlToRigidBodyList chk_enableBackfaceCollision val

	on chk_contact_overrideGlobals changed val do	
	(
		px_multiedit_methods.SetControlToRigidBodyList chk_contact_overrideGlobals val
		UpdateUI_ContactShellEnables()
	)
	on spn_contact_distance changed val do			px_multiedit_methods.SetControlToRigidBodyList spn_contact_distance val
	on spn_contact_restDepth changed val do			px_multiedit_methods.SetControlToRigidBodyList spn_contact_restDepth val	
	
	on rdo_advMode changed val do						px_multiedit_methods.SetControlToRigidBodyList rdo_AdvMode val
	on spn_velX changed val do								px_multiedit_methods.SetControlToRigidBodyList spn_velX val
	on spn_velY changed val do								px_multiedit_methods.SetControlToRigidBodyList spn_velY val
	on spn_velZ changed val do								px_multiedit_methods.SetControlToRigidBodyList spn_velZ val
	on spn_velSpeed changed val do						px_multiedit_methods.SetControlToRigidBodyList spn_velSpeed val
	on spn_spinX changed val do								px_multiedit_methods.SetControlToRigidBodyList spn_spinX val
	on spn_spinY changed val do								px_multiedit_methods.SetControlToRigidBodyList spn_spinY val
	on spn_spinZ changed val do								px_multiedit_methods.SetControlToRigidBodyList spn_spinZ val
	on spn_spinSpeed changed val do						px_multiedit_methods.SetControlToRigidBodyList spn_spinSpeed val

	on rdo_massCenterMode changed val do
	(
		px_multiedit_methods.SetControlToRigidBodyList rdo_massCenterMode val
		UpdateUI_MassCenterEnables()
	)
	on spn_massCenterX changed val do					px_multiedit_methods.SetControlToRigidBodyList spn_massCenterX val
	on spn_massCenterY changed val do					px_multiedit_methods.SetControlToRigidBodyList spn_massCenterY val
	on spn_massCenterZ changed val do					px_multiedit_methods.SetControlToRigidBodyList spn_massCenterZ val
	
	on spn_linDamping changed val do						px_multiedit_methods.SetControlToRigidBodyList spn_linDamping val
	on spn_angDamping changed val do					px_multiedit_methods.SetControlToRigidBodyList spn_angDamping val

	fn IsVisible =
	(
		(px_multiedit_methods.IsTabVisible()) and
			px_multiedit_methods.showRigidBodyRollouts
	)

	fn UpdateUI =
	(
		-- Basic default handling
		px_multiedit_methods.UpdateUI_DefaultHandling_RigidBody controlInfoList
		
		UpdateUI_ContactShellEnables()
		UpdateUI_MassCenterEnables()
		UpdateUI_StaticRBEnables()
	)

	on px_panel_multiedit_advanced open do
	(
		px_panel_multiedit_advanced.title = nvpxText.panelMultiAdvanced

		grp_simulationOverrides.text = nvpxText.panelMultiAdvancedSim
		chk_overSolverIter.text = nvpxText.panelMultiAdvancedSolverIter
		chk_enableBackfaceCollision.text = nvpxText.panelMultiAdvancedEnableBackfaceCollision
		
		grp_contactshell.text = nvpxText.panelMultiAdvancedContactShell
		chk_contact_overrideGlobals.text = nvpxText.panelMultiAdvancedContactShellOverride
		lbl_contact_distance.text = nvpxText.panelMultiAdvancedContactShellContactDistance
		lbl_contact_restDepth.text = nvpxText.panelMultiAdvancedContactShellRestDepth

		grp_initialMotion.text = nvpxText.panelMultiAdvancedIniMotion
		-- NOTE: rdo_advMode radio buttons already localized, in declaration
		lbl_initialVelocity.text = nvpxText.panelMultiAdvancedIniVel
		lbl_velX.text = nvpxText.panelMultiAdvancedX
		lbl_velY.text = nvpxText.panelMultiAdvancedY
		lbl_velZ.text = nvpxText.panelMultiAdvancedZ
		lbl_velSpeed.text = nvpxText.panelMultiAdvancedSpeed
		lbl_initialSpin.text = nvpxText.panelMultiAdvancedIniSpin
		lbl_spinX.text = nvpxText.panelMultiAdvancedX
		lbl_spinY.text = nvpxText.panelMultiAdvancedY
		lbl_spinZ.text = nvpxText.panelMultiAdvancedZ
		lbl_spinSpeed.text = nvpxText.panelMultiAdvancedSpeed
		
		grp_centerOfMass.text = nvpxText.panelMultiAdvancedCenterOfMass
		-- NOTE: rdo_massCenterMode radio buttons already localized, in declaration
		lbl_massCenterX.text = nvpxText.panelMultiAdvancedX
		lbl_massCenterY.text = nvpxText.panelMultiAdvancedY
		lbl_massCenterZ.text = nvpxText.panelMultiAdvancedZ

		grp_damping.text = nvpxText.panelMultiAdvancedDamping
		lbl_linDamping.text = nvpxText.panelMultiAdvancedLinear
		lbl_angDamping.text = nvpxText.panelMultiAdvancedAngular
		
		grp_simulationOverrides.height = 24 + (chk_enableBackfaceCollision.pos.y - grp_simulationOverrides.pos.y)
		grp_contactshell.height = 24 + (spn_contact_restDepth.pos.y - grp_contactshell.pos.y)
		grp_initialMotion.height = 24 + (spn_spinSpeed.pos.y - grp_initialMotion.pos.y)
		grp_centerOfMass.height = 24 + (spn_massCenterZ.pos.y - grp_centerOfMass.pos.y)
		grp_damping.height = 24 + (spn_angDamping.pos.y - grp_damping.pos.y)
		
		px_panel_multiedit_advanced.height += 8
	)

	on px_panel_multiedit_advanced help do (HelpSystem.ShowProductHelp 15012) --id_massfx_tools_edit
)


---------- ---------- ---------- ----------
-- Forces Rollout
---------- ---------- ---------- ----------

rollout px_panel_multiedit_forces "Forces" rolledUp:true category:7
(
	
	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC

	-- TO DO: No support to pick multiple forces
	-- TO DO: No support to display "<varies>" when selected objects have different forces
	-- TO DO: Only hacky support to display forces that have been deleted
	checkbox chk_UseGravity "Use World Gravity" across:1 align:#left offset:[0,4] checked:true
	Label lbl_forces "Scene Forces Applied:" align:#left offset:[5,4]
	ListBox lst_forces "" height:8
	PickButton btn_forcesAdd "Add" width:74 across:2 \
		filter:px_panel_multiedit_forces.ForcesAddFilter
	Button btn_forcesRemove "Remove" width:74
	
	-- IMPORTANT:  KEEP CONTROL DECLARATIONS / CONTROL INFO LIST IN SYNC


	local controlInfoList =
	#(
		(MultiEdit_ControlInfo_struct		controlItem:chk_UseGravity			paramIdent:#enableGravity),
		(MultiEdit_ControlInfo_struct		controlItem:lst_forces					paramIdent:#ForcesList),
		(MultiEdit_ControlInfo_struct		controlItem:btn_forcesAdd),
		(MultiEdit_ControlInfo_struct		controlItem:btn_forcesRemove)
	)

	fn GetForcesArray =
	(	-- SortedArray of forces, in same order as entries in the list box
		controlInfoList[2].paramValArray -- this is updated by the system
	)
	fn ForcesAddFilter item =  -- Used for Add button.  Must declared before the button
	(	-- returns true if item is a SpaceWarp Force, and not not already in the list
		local paramValArray = GetForcesArray()
		if (paramValArray!=undefined) and ((paramValArray.FindIndex item) > 0) then false
		else WSMSupportsForce item
	)
	
	fn IsVisible =
	(
		(px_multiedit_methods.IsTabVisible()) and
			px_multiedit_methods.showRigidBodyRollouts
	)

	fn UpdateUI =
	(
		-- Basic default handling
		px_multiedit_methods.UpdateUI_DefaultHandling_RigidBody controlInfoList
		
		local bakedVal = px_multiedit_methods.GetParamFromRigidBodyList #baked
		if (bakedVal!=undefined) do bakedVal = (bakedVal>0)  -- Bake modes other than zero indicate baked object
		
		chk_UseGravity.enabled = (bakedVal==false)
		lst_forces.enabled = (bakedVal==false)
		btn_forcesAdd.enabled = (bakedVal==false)
		local removeEnabled = false
		local paramValArray = GetForcesArray()
		if (paramValArray != undefined AND paramValArray.itemArray.count > 0) do removeEnabled = true
		btn_forcesRemove.enabled = removeEnabled 
	)

	on chk_UseGravity changed val do		px_multiedit_methods.SetControlToRigidBodyList chk_UseGravity val
	on btn_forcesAdd picked paramVal do
	(
		px_multiedit_methods.AddParamEntryToRigidBodyList #forcesList paramVal
		UpdateUI()
	)
	on btn_forcesRemove pressed do
	(
		local paramValArray = GetForcesArray()
		if (paramValArray!=undefined) do
		(
			local selIndex = lst_forces.selection
			if (selIndex != 0) do
			(
				local selForceName = lst_forces.items[selIndex]
				-- Since lst_forces is sorted alphabetically, and paramValArray is sorted on animHandle,
				-- need to find the paramValArray index that corresponds to the selected force name
				local arrayIndex = 0
				for i = 1 to paramValArray.itemArray.count do (
					local n = paramValArray.itemArray[i]
					if (isKindof n Node AND n.name == selForceName) do (
						arrayIndex = i
					)
				) 
				if (arrayIndex > 0) do
				(
					local paramVal = paramValArray.itemArray[arrayIndex]
					px_multiedit_methods.RemoveParamEntryFromRigidBodyList #forcesList paramVal
					UpdateUI()
				)
			)
		)
	)
	
	on px_panel_multiedit_forces open do
	(
		px_panel_multiedit_forces.title = nvpxText.panelMultiForces
		chk_UseGravity.text = nvpxText.panelMultiForcesGravity
		lbl_forces.text = nvpxText.panelMultiForcesList
		btn_forcesAdd.text = nvpxText.panelMultiForcesAdd
		btn_forcesRemove.text = nvpxText.panelMultiForcesRemove
	)

	on px_panel_multiedit_forces help do (HelpSystem.ShowProductHelp 15012) --id_massfx_tools_edit
)

---------- ---------- ---------- ----------
-- Multi-Edit Tab Methods
-- Helper methods, data, entry point to the tab
---------- ---------- ---------- ----------

struct px_multiedit_methods_struct
(
	---------- ----------
	-- Member data
	---------- ----------

	tabItem = undefined,				-- UI control for this tab
	tabIndex = undefined,				-- Index of this tab
	tabIndexCur = undefined,			-- Index of the currently visible tab
	tabRolloutContainer = undefined,	-- Rollout container for this tab
	tabRollouts = 
		#(
			px_panel_multiedit_selection,
			
			px_panel_multiedit_properties,
			
			px_panel_multiedit_materialList,
			px_panel_multiedit_material,
			
			px_panel_multiedit_mesh,
			px_panel_multiedit_sphereParams,
			px_panel_multiedit_boxParams,
			px_panel_multiedit_capsuleParams,
			px_panel_multiedit_convexParams,
			px_panel_multiedit_compositeParams,
			px_panel_multiedit_originalParams,
			px_panel_multiedit_customParams,
			
			px_panel_multiedit_forces,
			
			px_panel_multiedit_advanced,

			px_panel_multiedit_constraints_general,
			px_panel_multiedit_constraints_translationLimits,
			px_panel_multiedit_constraints_swingTwistLimits,
			px_panel_multiedit_constraints_spring,
			px_panel_multiedit_constraints_advanced
		),
	rolloutInfoList = undefined,  -- Objects of type MultiEdit_RolloutInfo_struct, for each rollout above
	controlInfoListSorted = undefined,
	showRigidBodyRollouts = false,
	showConstraintRollouts = false,


	---------- ----------
	-- Member methods, helpers
	---------- ----------

	fn Zero =
	(
		tabItem = undefined
		tabIndex = undefined
		tabIndexCur = undefined
		tabRolloutContainer = undefined
		tabRollouts = undefined
		rolloutInfoList = undefined
		controlInfoListSorted = undefined
	),

	-- Calculate a sort key integer, from a UI control
	fn ControlItemToKeyVal controlItem =
	(
		-- Control may be multiple windows, for example, spinner control plus edit box
		-- hwnd is an array of all window handles, use the first one
		(controlItem.hwnd)[1] as integer
	),

	-- Calculate a sort key struct, from a UI control
	fn ControlItemToKeyValStruct controlItem =
	(
		-- Control may be multiple windows, for example, spinner control plus edit box
		-- hwnd is an array of all window handles, use the first one
		KeyVal_struct keyVal:(ControlItemToKeyVal controlItem)
	),

	fn GetControlInfo controlItem =
	(
		if controlItem == undefined then undefined
		else
		(
			-- 1. Sort the list (if needed, just-in-time construction)
			if controlInfoListSorted==undefined do this.InitControlInfoList()

			-- 2. Search the list
			keyVal = (ControlItemToKeyValStruct controlItem)
			bsearch keyVal controlInfoListSorted KeyVal_struct.comparator
		)
	),
	
	fn SortedArrayToStringArray sortedArray =
	(
		local stringArray = SortedArray_Struct()
		for item in sortedArray.itemArray do
		(	-- TO DO: Hack, this assumes input is an ArrayParameter of Node objects
			if (item==undefined) then stringArray.AddItem nvpxText.panelMultiForcesDeletedItem
			-- if item is a node return it's name, else convert to string
			else if (isKindof item Node) then stringArray.AddItem item.name
			else append stringArray.AddItem (item as string)
		)
		stringArray.itemArray
	),
	
	-- Calculate an internal parameter value, from a UI control value.  Requires a controlInfo struct.
	fn ControlValToParamVal controlInfo controlVal =
	(
		if (controlInfo.controlValLookup==undefined) then (controlVal) else (controlInfo.controlValLookup.ControlToParamFn controlInfo.controlItem controlVal)
	),

	-- Calculate a UI control value, from an internal parameter value.  Requires a controlInfo struct.
	fn ParamValToControlVal controlInfo paramVal =
	(
		if ((classOf paramVal) == SortedArray_Struct) then SortedArrayToStringArray paramVal
		else if (controlInfo.controlValLookup==undefined) then (paramVal) else (controlInfo.controlValLookup.ParamToControlFn controlInfo.controlItem paramVal)
	),

	-- Apply a parameter value to all items in a list.
	fn SetParamToItems items paramIdent paramVal =
	(	-- Sets item.paramIdent=paramVal, for each item
		if paramVal != undefined do
		(
			local x
			for x in items do
			(
				setProperty x paramIdent paramVal
			)
		)
	),
	
	fn GetSortKey item =
	(	-- Numerical key for any kind of item, including scene objects
		if (item==undefined) then -1
		else if (isKindof item MaxWrapper) then (GetHandleByAnim item)
		else item  -- other types not yet supported
	),
	fn ParamValComparator a b =
	(	-- Note, comparator will be evaluated in separate context, need scope for GetSortKey()
		local keyA = px_multiedit_methods.GetSortKey a
		local keyB = px_multiedit_methods.GetSortKey b
		if keyA > keyB then 1
		else if keyA < keyB then -1
		else 0
	),
	fn AddParamEntryToItems items paramIdent paramVal =
	(	-- Appends paramVal to item.paramIdent list, for each item.  Assumes parameter is a list
		if paramVal != undefined do
		(
			local x
			for x in items do
			(
				appendIfUnique (getProperty x paramIdent) paramVal
			)
		)
	),
	fn RemoveParamEntryFromItems items paramIdent paramVal =
	(	-- Removes paramVal from item.paramIdent list, for each item.  Assumes parameter is a list
		-- Allowed to remove 'undefined', which indicates scene deleted nodes, in a node list
		local x, i
		for x in items do
		(
			local paramValArray = (getProperty x paramIdent)
			for i = paramValArray.count to 1 by -1 do
			(
				if paramValArray[i]==paramVal do deleteItem paramValArray i
			)
		)
	),

	fn InvokeFuncOnItems items funcOpVal funcArgVal =
	(
		for item in items do (funcOpVal item funcArgVal)
	),
	
	-- Fetch a value as returned by a given function, from items in a list.  If not all items agree on the value, returns undefined.
	fn GetFuncResultFromItems items funcOpVal funcArgVal =
	(	-- Gets funcResultVal=(funcOpVal item funcArgVal), for all items, or undefined if not all funcResultVal match
		local funcResultVal = undefined, match = true
		if items.count > 0 do
		(
			funcResultVal = funcOpVal items[1] funcArgVal
			for i = 2 to items.count while match do
				match = (funcResultVal==(funcOpVal items[i] funcArgVal))
		)
		if match then funcResultVal else undefined
	),
	
	fn GetParamArrayFromItems items paramIdent =  -- Helper for GetParamFromItems()
	(	-- Assumes an array parameter ... Returns SortedArray containing all entries in common among the items
		local paramValArray = SortedArray_Struct()

		paramValArray.init (getProperty items[1] paramIdent) comparator:ParamValComparator
		local match = true, commonCount = paramValArray.itemArray.count
		for i = 2 to items.count while match or (commonCount>0) do
		(
			local paramValArrayCur = (getProperty items[i] paramIdent)
			local startCount = commonCount, curCount = paramValArrayCur.count
			
			paramValArray.SetIntersection paramValArrayCur
			commonCount = paramValArray.itemArray.count
			if (commonCount!=startCount) or (commonCount!=curCount) then
				match = false
		)
		
		--if (not match) then paramEntries.AddItem #marker  -- add marker to indicate there are differences
		paramValArray  -- return value
	),
	fn GetParamPrimitiveFromItems items paramIdent =  -- Helper for GetParamFromItems()
	(	-- Assume non-array parameters ... Returns item.paramIdent, for all items, or undefined if not all match
		local paramVal = (getProperty items[1] paramIdent)

		local match = true
		for i = 2 to items.count while match do
			match = (paramVal==(getProperty items[i] paramIdent))
				
		if (not match) then paramVal = undefined
		paramVal  -- return value
	),
	
	-- Fetch a parameter value from items in a list.  If not all items agree on the value, returns undefined.
	fn GetParamFromItems items paramIdent =
	(	-- For array parameters ... Returns SortedArray containing all entries in common among the items
		-- For primitive parameters ... Returns item.paramIdent, for all items, or undefined if not all items match
		if items.count > 0 then
		(
			local paramVal = getProperty items[1] paramIdent
			if (isKindOf paramVal ArrayParameter) then
				GetParamArrayFromItems items paramIdent
			else
				GetParamPrimitiveFromItems items paramIdent
		)
		else undefined  -- default
	),

	-- Fetch a parameter value from a UI control.  Calculates the internal parameter value from the control value as appropriate.  
	fn GetParamFromControl controlInfo =
	(	-- Gets paramVal from a given UI control, for any control type
		local controlItem = controlInfo.controlItem
		local controlVal = undefined

		case (classOf controlItem) of
		(
		CheckBoxControl:		controlVal = (if controlItem.triState!=2 then controlItem.state else undefined)
		CheckButtonControl:		controlVal = (if controlItem.enabled then controlItem.state else undefined)
		RadioControl:			controlVal = (if controlItem.state>0 then controlItem.state else undefined)
		SpinnerControl:			controlVal = (if controlItem.indeterminate==false then controlItem.value else undefined)
		ComboBoxControl:		controlVal = (if controlItem.selection>0 then controlItem.selection else undefined)
		ButtonControl:			controlVal = undefined
		)

		ControlValToParamVal controlInfo controlVal  -- If lookup is undefined, returns input unchanged
	),
	
	-- Apply a parameter value to a UI control.  Calculates the control value from the internal parameter value as appropriate.  
	fn SetParamToControl controlInfo paramVal =
	(	-- Sets paramVal to a given UI control, for any control type.  Assumes input paramVal is correct type for controlItem
		local controlItem = controlInfo.controlItem
		local controlVal =  ParamValToControlVal controlInfo paramVal  -- If lookup is undefined, returns input unchanged

		case (classOf controlItem) of
		(
		CheckBoxControl:
			if controlVal == undefined then controlItem.triState = 2  -- Indeterminate tri-state
			else controlItem.state = controlVal
		CheckButtonControl:
			if controlVal == undefined then (controlItem.state = false; controlItem.enabled = false)  -- Indeterminate tri-state
			else (controlItem.state = controlVal; controlItem.enabled = true)
		RadioControl:
			if controlVal == undefined then controlItem.state = 0  -- Indeterminate tri-state
			else controlItem.state = controlVal
		SpinnerControl:
			if controlVal == undefined then controlItem.indeterminate = true  -- Indeterminate tri-state
			else controlItem.value = controlVal
		ComboBoxControl:
			if controlVal == undefined then controlItem.selection = 0  -- Indeterminate tri-state
			else controlItem.selection = controlVal
		ButtonControl:
			if controlVal == undefined then controlItem.enabled = false  -- Indeterminate tri-state
			else controlItem.enabled = true
		ListBoxControl:
			(
				--local sel = controlItem.selection
				controlItem.items = controlVal --; controlItem.selection = sel
				controlInfo.paramValArray = paramVal
			)
		)
	),

	-- Fetch a parameter value from a UI control.  Calculates the internal parameter value from the control value as appropriate.
	-- Requires on a controlInfo, fetched using the UI control item.
	fn GetParamFromControlItem controlItem =
	(
		local controlInfo = GetControlInfo controlItem
		GetParamFromControl controlInfo
	),
	
	fn AddParamEntryToRigidBodyList paramIdent paramVal =
	(
		AddParamEntryToItems pxSelectedRigidBodyMods paramIdent paramVal
	),
	fn RemoveParamEntryFromRigidBodyList paramIdent paramVal =
	(
		RemoveParamEntryFromItems pxSelectedRigidBodyMods paramIdent paramVal
	),

	fn InvokeFuncOnRigidBodyList funcOpVal funcArgVal =
	(
		undo "PhysX Multi-Object Editor" on
		(
			InvokeFuncOnItems pxSelectedRigidBodyMods funcOpVal funcArgVal
		)
	),

	-- Fetch a value as returned by a given function, from currently selected Rigid Body objects.  If not all items agree on the value, returns undefined.
	fn GetFuncResultFromRigidBodyList funcOpVal funcArgVal =
	(
		GetFuncResultFromItems pxSelectedRigidBodyMods funcOpVal funcArgVal
	),
	
	-- Fetch a parameter value from currently selected Rigid Body objects.  If not all items agree on the value, returns undefined.
	fn GetParamFromRigidBodyList paramIdent =
	(
		GetParamFromItems pxSelectedRigidBodyMods paramIdent
	),
	
	-- Apply a parameter value to currently selected Rigid Body objects.
	fn SetParamToRigidBodyList paramIdent paramVal =
	(
		undo "PhysX Multi-Object Editor" on
		(
			SetParamToItems pxSelectedRigidBodyMods paramIdent paramVal
		)
	),

	-- Apply a parameter value to given objects. Calculates the internal parameter value from the control value as appropriate.
	-- Requires on a controlInfo, fetched using the UI control item.
	fn SetControlToItems items controlItem controlVal =
	(	
		-- If any Rigid Body objects selected, proceed
		if items.count > 0 do
		(
			if controlVal != undefined do
			(
				-- Get the Control Info struct for this UI control
				local controlInfo = GetControlInfo controlItem

				-- Get the Rigid Body parameter for the UI info
				local paramIdent = controlInfo.paramIdent

				if paramIdent != undefined do
				( 
					-- Lookup the corresponding paramVal for this controlVal
					local paramVal = ControlValToParamVal controlInfo controlVal  -- If lookup is undefined, returns input unchanged
					
					SetParamToItems items paramIdent paramVal
				)
			)
			-- TO DO: Else, add debug assert, should never happen
		)
	),
	-- Apply a parameter value to currently selected Rigid Body objects. Calculates the internal parameter value from the control value as appropriate.
	-- Requires on a controlInfo, fetched using the UI control item.
	fn SetControlToRigidBodyList controlItem controlVal =
	(	
		undo "PhysX Multi-Object Editor" on
		(
			SetControlToItems pxSelectedRigidBodyMods controlItem controlVal
		)
	),

	-- Fetch a parameter value from currently selected Constraint objects.  If not all items agree on the value, returns undefined.
	fn GetParamFromConstraintList paramIdent =
	(
		GetParamFromItems pxSelectedConstraints paramIdent
	),

	-- Apply a parameter value to currently selected Constraint objects.
	fn SetParamToConstraintList paramIdent paramVal =
	(
		undo "PhysX Multi-Object Editor" on
		(
			SetParamToItems pxSelectedConstraints paramIdent paramVal
		)
	),

	-- Apply a parameter value to currently selected Constraint objects. Calculates the internal parameter value from the control value as appropriate.
	-- Requires on a controlInfo, fetched using the UI control item.
	fn SetControlToConstraintList controlItem controlVal =
	(	
		undo "PhysX Multi-Object Editor" on
		(
			SetControlToItems pxSelectedConstraints controlItem controlVal
		)
	),	

	---------- ----------
	-- Member methods, UI handling
	---------- ----------

	fn IsTabVisible =
	(
		(tabIndex != undefined) and (tabIndex == tabIndexCur)
	),

	-- Create rollout configuration data or load from ini file
	fn GetRolloutInfoList =
	(
		if (this.rolloutInfoList == undefined) then 
		(
			-- create a list of RolloutInfo objects, using a for-loop collection on the rollout list
			this.rolloutInfoList = for r in this.tabRollouts collect (PxRolloutInfo_struct rolloutVal:r)

			-- load the state data from ini file, for each entry
			PxLoadRolloutInfoList this.rolloutInfoList
		)
		this.rolloutInfoList  -- return value
	),
	
	-- Hide or show rollouts, according to their current visibility
	fn UpdateUI_TabRollouts =
	(
		this.showRigidBodyRollouts = ((pxSelectedConstraints.count==0) and (pxSelectedRigidBodyMods.count>0))
		this.showConstraintRollouts = ((pxSelectedConstraints.count > 0) and (pxSelectedRigidBodyMods.count==0))
		
		if tabRolloutContainer!=undefined do
		(
			local rolloutInfoList = this.GetRolloutInfoList()
			PxUpdateRolloutInfoList rolloutInfoList  -- store each rollout's current state information, before removing any rollouts
			
			for rolloutInfoVal in rolloutInfoList do
			(
				local rolloutVal = rolloutInfoVal.rolloutVal
				if (IsTabVisible()) and (rolloutVal.IsVisible()) then  -- rollout should be displayed ...
				(
					if not rolloutVal.isDisplayed then -- ... but not currently displayed, add it
						addSubRollout tabRolloutContainer rolloutVal rolledUp:(rolloutInfoVal.rolloutRolledUp)
				)
				else  -- rollout should not be displayed ...
				(
					if rolloutVal.isDisplayed then  -- ... but is currently displayed, remove it
						removeSubRollout tabRolloutContainer rolloutVal
				)

			)
		)
		this.InitControlInfoList()
	),

	-- Update UI default handling, usable by any rollout which pertains to the given items.
	-- Updates all controls in the rollout's controlInfo list.
	fn UpdateUI_DefaultHandling_Items items controlInfoList updateEnables:true =
	(
		local enableUI = (items.count > 0)
		for controlInfo in controlInfoList do
		(
			if updateEnables do controlInfo.controlItem.enabled = enableUI

			if controlInfo.paramIdent != undefined do
			(
				local paramVal = this.GetParamFromItems items controlInfo.paramIdent
				this.SetParamToControl controlInfo paramVal
			)
		)
	),

	-- Update UI default handling, usable by any rollout which pertains to Rigid Body objects.
	-- Updates all controls in the rollout's controlInfo list.
	fn UpdateUI_DefaultHandling_RigidBody controlInfoList updateEnables:true =
	(
		UpdateUI_DefaultHandling_Items pxSelectedRigidBodyMods controlInfoList updateEnabled:updateEnables
	),

	-- Update UI default handling, usable by any rollout which pertains to Constraint objects.
	-- Updates all controls in the rollout's controlInfo list.
	fn UpdateUI_DefaultHandling_Constraint controlInfoList updateEnables:true =
	(
		UpdateUI_DefaultHandling_Items pxSelectedConstraints controlInfoList updateEnables:updateEnables
	),
	
	-- Update UI handler for entire tab.  Updates each visible rollout.
	fn UpdateUI isForceUpdate:false =
	(
		if IsTabVisible() do
			for rolloutVal in tabRollouts do
				if rolloutVal.isDisplayed do
					rolloutVal.UpdateUI isForceUpdate:isForceUpdate
	),


	fn UpdateUI_MatRollout = 
	(
		if IsTabVisible() do
		(
			if px_panel_multiedit_material.isDisplayed do
				px_panel_multiedit_material.UpdateUI isForceUpdate:true
		)
	),

	fn UpdateUI_MatListRollout = 
	(
		if IsTabVisible() do
		(
			if px_panel_multiedit_materialList.isDisplayed do
				px_panel_multiedit_materialList.UpdateUI isForceUpdate:true
		)
	),
	
	fn UpdateUI_MeshRollout = 
	(
		if IsTabVisible() do
		(
			if px_panel_multiedit_mesh.isDisplayed do
				px_panel_multiedit_mesh.UpdateUI isForceUpdate:true
		)
	),

	fn UpdateUI_ConstraintLimitRollouts = 
	(
		if IsTabVisible() do
		(
			if px_panel_multiedit_constraints_translationLimits.isDisplayed do
				px_panel_multiedit_constraints_translationLimits.UpdateUI isForceUpdate:true
			if px_panel_multiedit_constraints_swingTwistLimits.isDisplayed do
				px_panel_multiedit_constraints_swingTwistLimits.UpdateUI isForceUpdate:true
		)
	),	

	fn OnSelectionSetChange =
	(
		this.UpdateUI isForceUpdate:true
	),

	fn OnUndo =
	(
		UpdateUI()
	),

	fn OnSwitchTabs tabContainer tabRolloutContainer =
	(
		this.tabRolloutContainer = tabRolloutContainer
		this.tabIndexCur = tabContainer.SelectedIndex

		px_UpdateSelectedRigidBodyLists() -- Hack, force an update of the selected object list
		UpdateUI_TabRollouts()
		UpdateUI isForceUpdate:true
	),


	---------- ----------
	-- Member methods, constructor / destructor
	---------- ----------

	fn InitControlInfoList =
	(
		-- Initialize the control list, and create the sorted list
		
		-- 1. Create the master list of controls for all rollouts
		controlInfoListSorted = #()
		for rolloutVal in tabRollouts do
			if rolloutVal.isDisplayed do
				join controlInfoListSorted rolloutVal.controlInfoList

		-- 2. For each entry, define a key based on the HWND of the control
		for controlInfo in controlInfoListSorted do controlInfo.keyVal = (ControlItemToKeyVal controlInfo.controlItem)

		-- 3. Sort the master list according to key
		qsort controlInfoListSorted KeyVal_Struct.comparator
	),

	-- Host dialog is opened
	fn OnOpen tabItem tabIndex =
	(
		this.tabItem = tabItem
		this.tabIndex = tabIndex
		
		this.controlInfoListSorted = undefined  -- Will be initialized later, just-in-time
		this.rolloutInfoList = undefined -- Will be initialized later, just-in-time
	),

	-- Host dialog is closed
	fn OnClose =
	(
		PxSaveRolloutInfoList this.rolloutInfoList
	)	
	
) -- end struct px_multiedit_methods_struct


---------- ---------- ---------- ----------
-- Multi-Edit Tab Singleton
---------- ---------- ---------- ----------

px_multiedit_methods  = px_multiedit_methods_struct()
px_tab_multiedit = px_multiedit_methods -- Name overload

-------BEGIN-SIGNATURE-----
-- 4wYAADCCBt8GCSqGSIb3DQEHAqCCBtAwggbMAgEBMQ8wDQYJKoZIhvcNAQELBQAw
-- CwYJKoZIhvcNAQcBoIIE3jCCBNowggPCoAMCAQICEDUAFkMQxqI9PltZ2eUG16Ew
-- DQYJKoZIhvcNAQELBQAwgYQxCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRl
-- YyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1
-- MDMGA1UEAxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
-- IC0gRzIwHhcNMTkwNjI1MDAwMDAwWhcNMjAwODA3MjM1OTU5WjCBijELMAkGA1UE
-- BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExEzARBgNVBAcMClNhbiBSYWZhZWwx
-- FzAVBgNVBAoMDkF1dG9kZXNrLCBJbmMuMR8wHQYDVQQLDBZEZXNpZ24gU29sdXRp
-- b25zIEdyb3VwMRcwFQYDVQQDDA5BdXRvZGVzaywgSW5jLjCCASIwDQYJKoZIhvcN
-- AQEBBQADggEPADCCAQoCggEBAMsptjSEm+HPve6+DClr+K4CgrtrONjtHxHBwTMC
-- mrwF9bnsdMiSgvYigTKk858TlqVs7GiBVLD3SaSZqfSXOv7L55i965L+wIx0EZxX
-- xDzbyLh1rLSSNWO8oTDIKnPsiwo5x7CHRUi/eAICOvLmz7Rzi+becd1j/JPNWe5t
-- vum0GL/8G4vYICrhCycizGIuv3QFqv0YPM75Pd2NP0V4W87XPeTrj+qQoRKMztJ4
-- WNDgLgT4LbMBIZyluU8iwXNyWQ8FC2ya3iJyy0EhZhAB2H7oMrAcV1VJJqwZcZQU
-- XMJTD+tuCqKqJ1ftv1f0JVW2AADnHgvaB6E6Y9yR/jnn4zECAwEAAaOCAT4wggE6
-- MAkGA1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMD
-- MGEGA1UdIARaMFgwVgYGZ4EMAQQBMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5z
-- eW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5zeW1jYi5jb20v
-- cnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIw
-- IKAeoByGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUFBwEBBEsw
-- STAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5bWNkLmNvbTAmBggrBgEFBQcwAoYa
-- aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcnQwDQYJKoZIhvcNAQELBQADggEBADo7
-- 6cASiVbzkjsADk5MsC3++cj9EjWeiuq+zzKbe55p6jBNphsqLUvMw+Z9r2MpxTEs
-- c//MNUXidFsslWvWAUeOdtytNfhdyXfENX3baBPWHhW1zvbOPHQLyz8LmR1bNe9f
-- R1SLAezJaGzeuaY/Cog32Jh4qDyLSzx87tRUJI2Ro5BLA5+ELiY21SDZ7CP9ptbU
-- CDROdHY5jk/WeNh+3gLHeikJSM9/FPszQwVc9mjbVEW0PSl1cCLYEXu4T0o09ejX
-- NaQPg10POH7FequNcKw50L63feYRStDf6GlO4kNXKFHIy+LPdLaSdCQL2/oi3edV
-- MdpL4F7yw1zQBzShYMoxggHFMIIBwQIBATCBmTCBhDELMAkGA1UEBhMCVVMxHTAb
-- BgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBU
-- cnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
-- b2RlIFNpZ25pbmcgQ0EgLSBHMgIQNQAWQxDGoj0+W1nZ5QbXoTANBgkqhkiG9w0B
-- AQsFADANBgkqhkiG9w0BAQEFAASCAQBNgwFMvc1SNjIK7CPaY9C0rKt+ZdThM98M
-- wPO45ptlXNIwFOhz0beikt+LiyRvOk0X67MfJY4ml4PFfDDuSIMsk+exL3T+azBa
-- 3JV5izR5mWnwzO6fnouxs1OB8PmOrfi+PPbc9v5tGylOJUiul2lenUhe43JvQZYR
-- c5aHhqD9GzgPDyZ0Azfyi19Oxyk2MRMLqRq9/X/OFkh5co09+MJRoNDgaUAFAufK
-- l2y1Oc2SfalaIN2oHfaXfNjkVoYP+gLp9vKK4Il3QlUfjUH6opJWWokFPOzv8LN6
-- vKMO/CRhbaih9hvtG4P6VPBg8bs1dKoELFp+44GQLXxpep6cRPng
-- -----END-SIGNATURE-----